namespace org.owasp.appsensor.filters
{
    using System;
    using System.Collections.Generic;
    using System.Web;
    using System.Web.SessionState;
    using org.owasp.appsensor.errors;

    /**
     * This class is a servlet filter that can be configured in the web.xml to 
     * detect if the user agent associated with a given session id changes mid-session.  
     * This could point to an attacker trying to reuse a session id from another machine.
     * 
     * The configuration in web.xml should be setup like this.
     * 
     * <filter>
     *    	<filter-name>UserAgentChangeDetectionFilter</filter-name>
     *    	<filter-class>
     *			org.owasp.appsensor.filters.UserAgentChangeDetectionFilter
     *		</filter-class>
     * 	</filter>
     * 	<filter-mapping>
     *	    <filter-name>UserAgentChangeDetectionFilter</filter-name>
     *	    <url-pattern>/*</url-pattern>
     * 	</filter-mapping>
     * 
     */
    public class UserAgentChangeDetectionFilter : IHttpModule
    {

        /** Map that holds mappings of session id's to user agent's. */
        //jtm - 5/23/2011 - change to concurrent map to address http://code.google.com/p/appsensor/issues/detail?id=6
        private static Dictionary<String, String> sessionUserAgentMap = new Dictionary<String, String>();

        public void Dispose()
        {
        }

        public void Init(HttpApplication context)
        {
            HttpSessionState session = context.Session;
            String userAgent = context.Request.Headers["user-agent"];

            //check if we can proceed
            if (session != null && session.SessionID != null && userAgent != null)
            {
                String sessionId = session.SessionID;

                //if session id is new to tracking - just add it to map
                if (!sessionUserAgentMap.ContainsKey(sessionId))
                {
                    if (!sessionUserAgentMap.ContainsKey(sessionId)) sessionUserAgentMap.Add(sessionId, userAgent);
                }
                else
                {
                    //session id is already in map - check if userAgent matches
                    String existingUserAgent = sessionUserAgentMap[sessionId];

                    //if there is a discrepancy, report the exception
                    if (!userAgent.Equals(existingUserAgent))
                    {
                        new AppSensorException("SE6", "AppSensorUser Message SE6",
                                "User agent for session has changed - original user agent [" + existingUserAgent + "]" +
                                " / new user agent [" + userAgent + "] for session id [" + sessionId + "]");
                    }
                }
            }
        }
    }
}